<?php

class Installer
{
    public static function checkEnvironment()
    {
        $phpVersion = PHP_VERSION;
        $phpOk = version_compare($phpVersion, "7.1.0", ">=") && version_compare($phpVersion, "7.2.0", "<");

        $checks = array(
            array(
                "name" => "PHP 版本",
                "required" => "7.1.x",
                "current" => $phpVersion,
                "passed" => $phpOk,
                "tips" => $phpOk ? "通过" : "必须使用 PHP 7.1"
            ),
            array(
                "name" => "PDO SQLite 扩展",
                "required" => "已开启",
                "current" => extension_loaded("pdo_sqlite") ? "已开启" : "未开启",
                "passed" => extension_loaded("pdo_sqlite"),
                "tips" => extension_loaded("pdo_sqlite") ? "通过" : "请开启 pdo_sqlite"
            ),
            array(
                "name" => "includes/data 可写",
                "required" => "可写",
                "current" => ensure_dir(DATA_PATH) ? "可写" : "不可写",
                "passed" => ensure_dir(DATA_PATH),
                "tips" => ensure_dir(DATA_PATH) ? "通过" : "请检查目录权限"
            ),
            array(
                "name" => "install 目录可写",
                "required" => "可写",
                "current" => is_writable(ROOT_PATH . "/install") ? "可写" : "不可写",
                "passed" => is_writable(ROOT_PATH . "/install"),
                "tips" => is_writable(ROOT_PATH . "/install") ? "通过" : "请检查目录权限"
            ),
            array(
                "name" => "bak 目录可写",
                "required" => "可写",
                "current" => ensure_dir(UPLOAD_DIR) ? "可写" : "不可写",
                "passed" => ensure_dir(UPLOAD_DIR),
                "tips" => ensure_dir(UPLOAD_DIR) ? "通过" : "请检查目录权限"
            )
        );

        $passed = true;
        foreach ($checks as $item) {
            if (!$item["passed"]) {
                $passed = false;
                break;
            }
        }

        return array(
            "passed" => $passed,
            "checks" => $checks
        );
    }

    public static function install($username, $password, $siteName)
    {
        if (!preg_match("/^[a-zA-Z0-9_]{3,20}$/", $username)) {
            throw new Exception("管理员账号只能包含字母、数字、下划线，长度 3-20 位");
        }
        if (strlen($password) < 6) {
            throw new Exception("管理员密码至少 6 位");
        }
        if ($siteName === "") {
            $siteName = "我的导航页";
        }

        $env = self::checkEnvironment();
        if (!$env["passed"]) {
            throw new Exception("环境检测未通过，请先修复后再安装");
        }

        if (file_exists(INSTALL_LOCK)) {
            throw new Exception("系统已安装，如需重装请先删除 install/install.lock");
        }

        if (!ensure_dir(DATA_PATH)) {
            throw new Exception("无法创建 includes/data 目录");
        }

        if (!ensure_dir(UPLOAD_DIR)) {
            throw new Exception("无法创建 bak 目录");
        }

        if (file_exists(DB_FILE) && !@unlink(DB_FILE)) {
            throw new Exception("旧数据库文件无法删除，请检查权限");
        }

        $pdo = new PDO("sqlite:" . DB_FILE);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        try {
            $pdo->beginTransaction();

            $pdo->exec("CREATE TABLE IF NOT EXISTS admins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT NOT NULL UNIQUE,
                password TEXT NOT NULL,
                created_at TEXT NOT NULL,
                updated_at TEXT NOT NULL
            )");

            $pdo->exec("CREATE TABLE IF NOT EXISTS settings (
                key_name TEXT PRIMARY KEY,
                value TEXT NOT NULL
            )");

            $pdo->exec("CREATE TABLE IF NOT EXISTS nav_items (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                url TEXT NOT NULL,
                icon TEXT NOT NULL DEFAULT '',
                icon_type TEXT NOT NULL DEFAULT 'font',
                description TEXT NOT NULL DEFAULT '',
                sort_order INTEGER NOT NULL DEFAULT 0,
                is_visible INTEGER NOT NULL DEFAULT 1,
                created_at TEXT NOT NULL,
                updated_at TEXT NOT NULL
            )");

            $pdo->exec("CREATE TABLE IF NOT EXISTS visit_logs (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                event_type TEXT NOT NULL,
                nav_item_id INTEGER NOT NULL DEFAULT 0,
                path TEXT NOT NULL DEFAULT '',
                target_url TEXT NOT NULL DEFAULT '',
                ip TEXT NOT NULL DEFAULT '',
                user_agent TEXT NOT NULL DEFAULT '',
                referer TEXT NOT NULL DEFAULT '',
                visit_date TEXT NOT NULL,
                created_at TEXT NOT NULL
            )");
            $pdo->exec("CREATE INDEX IF NOT EXISTS idx_visit_logs_date_type ON visit_logs(visit_date, event_type)");
            $pdo->exec("CREATE INDEX IF NOT EXISTS idx_visit_logs_nav ON visit_logs(nav_item_id)");

            $now = date("Y-m-d H:i:s");

            $stmtAdmin = $pdo->prepare("INSERT INTO admins (username, password, created_at, updated_at) VALUES (:username, :password, :created_at, :updated_at)");
            $stmtAdmin->execute(array(
                "username" => $username,
                "password" => password_hash($password, PASSWORD_DEFAULT),
                "created_at" => $now,
                "updated_at" => $now
            ));

            $settings = array(
                "site_name" => $siteName,
                "site_subtitle" => "轻松管理你的常用网站",
                "site_description" => "个人导航页系统",
                "theme_template" => "style2",
                "copyright_text" => "Copyright © " . date("Y"),
                "snow_enabled" => "0",
                "qq_number" => "",
                "qq_group" => ""
            );

            $stmtSetting = $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (:key_name, :value)");
            foreach ($settings as $key => $value) {
                $stmtSetting->execute(array("key_name" => $key, "value" => $value));
            }

            $sampleLinks = array(
                array("OpenAI", "https://openai.com", "fa-solid fa-robot", "AI 工具平台", 100),
                array("GitHub", "https://github.com", "fa-brands fa-github", "代码托管", 90),
                array("掘金", "https://juejin.cn", "fa-solid fa-code", "技术社区", 80),
                array("Bilibili", "https://www.bilibili.com", "fa-solid fa-play", "视频平台", 70)
            );

            $stmtLink = $pdo->prepare("INSERT INTO nav_items (title, url, icon, icon_type, description, sort_order, is_visible, created_at, updated_at)
                VALUES (:title, :url, :icon, :icon_type, :description, :sort_order, 1, :created_at, :updated_at)");

            foreach ($sampleLinks as $item) {
                $stmtLink->execute(array(
                    "title" => $item[0],
                    "url" => $item[1],
                    "icon" => $item[2],
                    "icon_type" => "font",
                    "description" => $item[3],
                    "sort_order" => $item[4],
                    "created_at" => $now,
                    "updated_at" => $now
                ));
            }

            $pdo->commit();
        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            throw new Exception("安装失败：" . $e->getMessage());
        }

        $lockContent = "installed_at=" . date("c") . "\n";
        if (@file_put_contents(INSTALL_LOCK, $lockContent) === false) {
            throw new Exception("安装成功但写入 install.lock 失败，请检查 install 目录权限");
        }

        return true;
    }
}
